package main

import (
	"ChristmasPass/signature"
	"bytes"
	"github.com/gin-gonic/gin"
	"github.com/liyue201/goqr"
	"github.com/thanhpk/randstr"
	"image"
	_ "image/jpeg"
	_ "image/png"
	"io"
	"net/http"
)

func main() {
	router := gin.Default()
	correctPath := "/" + randstr.String(64)
	samplePassCorrect := "/samplePassCorrect"
	defaultMsg := "Mustermann<<Max<Anton<<<26.05.1996"

	router.LoadHTMLGlob("templates/*")

	router.GET("/", func(context *gin.Context) {
		_, hasError := context.GetQuery("error")
		_, isSamplePass := context.GetQuery("sample")
		context.HTML(http.StatusOK, "index.html", gin.H{
			"Error":      hasError,
			"NotOwnPass": isSamplePass,
		})
	})

	router.GET(correctPath, func(context *gin.Context) {
		context.File("static/flag.png")
	})

	router.GET(samplePassCorrect, func(context *gin.Context) {
		context.JSON(http.StatusOK, gin.H{
			"message": "Yes, this is the pass you found, but you need your own!",
		})
	})

	router.POST("/passcheck", func(context *gin.Context) {
		file, _, err := context.Request.FormFile("upload")
		defer file.Close()
		if err != nil {
			context.Redirect(http.StatusFound, "/?error")
			return
		}

		buf := bytes.NewBuffer(nil)
		_, err = io.Copy(buf, file)
		if err != nil {
			context.Redirect(http.StatusFound, "/?error")
			return
		}

		img, _, err := image.Decode(bytes.NewReader(buf.Bytes()))
		if err != nil {
			context.Redirect(http.StatusFound, "/?error")
			return
		}

		codes, err := goqr.Recognize(img)
		if err != nil {
			context.Redirect(http.StatusFound, "/?error")
			return
		}

		for _, qrCode := range codes {
			ok, msg := signature.CheckSignatureString(string(qrCode.Payload))
			if ok && msg != defaultMsg {
				context.Redirect(http.StatusFound, correctPath)
				return
			} else if ok {
				context.Redirect(http.StatusFound, "/?sample")
				return
			}
		}

		context.Redirect(http.StatusFound, "/?error")
	})

	router.Run(":8080")
}
